<?php
// Защита от прямого доступа
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

// 1. Подключение стилей и скриптов Flatpickr
add_action('wp_enqueue_scripts', 'frontpad_enqueue_datepicker');
function frontpad_enqueue_datepicker() {
    if (is_checkout()) {
        // Стили Flatpickr
        wp_enqueue_style('flatpickr-css', 'https://cdn.jsdelivr.net/npm/flatpickr/dist/flatpickr.min.css');
        // Скрипт Flatpickr
        wp_enqueue_script('flatpickr-js', 'https://cdn.jsdelivr.net/npm/flatpickr', array('jquery'), null, true);
        // Русский язык
        wp_enqueue_script('flatpickr-ru', 'https://npmcdn.com/flatpickr/dist/l10n/ru.js', array('flatpickr-js'), null, true);
    }
}

// 2. Добавляем поле выбора времени (тип text, чтобы не вылезал родной календарь)
add_action( 'woocommerce_after_order_notes', 'add_preorder_datetime_field' );
function add_preorder_datetime_field( $checkout ) {
    echo '<div id="preorder_datetime_field"><h3>' . __('Время предзаказа') . '</h3>';

    woocommerce_form_field( 'preorder_datetime', array(
        'type'        => 'text', 
        'class'       => array('form-row-wide', 'frontpad-flatpickr-field'),
        'label'       => __('Выберите дату и время доставки'),
        'required'    => false,
        'placeholder' => __('Нажмите для выбора даты и времени'),
        'custom_attributes' => array(
            'readonly' => 'readonly', // Запрещаем ввод с клавиатуры
            'style'    => 'background-color: white; cursor: pointer;'
        ),
    ), $checkout->get_value( 'preorder_datetime' ));

    echo '</div>';
}

// 3. Валидация поля (Серверная проверка)
add_action('woocommerce_checkout_process', 'validate_preorder_datetime_field');
function validate_preorder_datetime_field() {
    if ( ! empty( $_POST['preorder_datetime'] ) ) {
        $preorder_datetime = sanitize_text_field( $_POST['preorder_datetime'] );
        // Flatpickr отдает формат Y-m-d H:i, PHP его прекрасно понимает
        $selected_time = strtotime( $preorder_datetime );
        $current_time = current_time('timestamp');

        if ( $selected_time === false || $selected_time < $current_time ) {
            wc_add_notice( __('Время предзаказа не может быть в прошлом.'), 'error' );
            return;
        }

        $hour = (int) date('H', $selected_time);
        $minute = (int) date('i', $selected_time);

        // Рабочее время с 10:00 до 21:00
        if ( $hour < 10 || $hour > 21 || ($hour == 21 && $minute > 0) ) {
            wc_add_notice( __('Время предзаказа должно быть с 10:00 до 21:00.'), 'error' );
        }
    }
}

// 4. Сохранение времени в заказ
add_action( 'woocommerce_checkout_update_order_meta', 'save_preorder_datetime_field' );
function save_preorder_datetime_field( $order_id ) {
    if ( ! empty( $_POST['preorder_datetime'] ) ) {
        $datetime = sanitize_text_field( $_POST['preorder_datetime'] );
        // Добавляем секунды :00 для совместимости с SQL форматом
        $datetime = $datetime . ':00';
        update_post_meta( $order_id, 'pre_order_time', $datetime );
    }
}

// 5. Отображение на странице "Спасибо"
add_action( 'woocommerce_thankyou', 'display_preorder_time_on_thankyou', 10, 1 );
function display_preorder_time_on_thankyou( $order_id ) {
    if ( ! $order_id ) return;
    $pre_order_time = get_post_meta( $order_id, 'pre_order_time', true );
    if ( ! empty( $pre_order_time ) ) {
        $formatted_time = date_i18n( 'd F Y, H:i', strtotime( $pre_order_time ) );
        echo '<div style="margin: 20px 0; padding: 15px; background: #eef; border-left: 4px solid #2196F3;">';
        echo '<strong>' . __('Время предзаказа:') . '</strong> ' . $formatted_time;
        echo '</div>';
    }
}

// 6. Отображение в письмах
add_action( 'woocommerce_email_order_meta', 'add_preorder_time_to_admin_emails', 10, 3 );
function add_preorder_time_to_admin_emails( $order, $sent_to_admin, $plain_text ) {
    if ( $sent_to_admin ) {
        $order_id = $order->get_id();
        $pre_order_time = get_post_meta( $order_id, 'pre_order_time', true );
        if ( ! empty( $pre_order_time ) ) {
            $formatted_time = date_i18n( 'd F Y, H:i', strtotime( $pre_order_time ) );
            if ( $plain_text ) {
                echo "\n" . __('Время предзаказа') . ": " . $formatted_time . "\n";
            } else {
                echo '<p><strong>' . __('Время предзаказа') . ':</strong> ' . $formatted_time . '</p>';
            }
        }
    }
}

// 7. JS инициализация календаря (Logic)
add_action('wp_footer', 'frontpad_init_flatpickr_js');
function frontpad_init_flatpickr_js() {
    if ( ! is_checkout() ) return;
    ?>
    <script type="text/javascript">
        jQuery(document).ready(function($) {
            var waitForFlatpickr = setInterval(function() {
                if (typeof flatpickr !== 'undefined') {
                    clearInterval(waitForFlatpickr);
                    initCalendar();
                }
            }, 100);

            function initCalendar() {
                var $field = $('input[name="preorder_datetime"]');
                if (!$field.length) return;

                var startHour = 10;
                var endHour = 21;

                flatpickr("input[name='preorder_datetime']", {
                    enableTime: true,
                    dateFormat: "Y-m-d H:i",
                    time_24hr: true,
                    locale: "ru",
                    minDate: "today",
                    maxDate: new Date().fp_incr(365),
                    disableMobile: "true", // Принудительно используем Flatpickr даже на телефоне
                    defaultHour: startHour,
                    defaultMinute: 0,
                    
                    onOpen: function(selectedDates, dateStr, instance) {
                        updateMinTime(instance);
                    },
                    onChange: function(selectedDates, dateStr, instance) {
                        updateMinTime(instance);
                    }
                });

                function updateMinTime(instance) {
                    var now = new Date();
                    var selectedDate = instance.selectedDates[0] || now;
                    var isToday = selectedDate.getDate() === now.getDate() && 
                                  selectedDate.getMonth() === now.getMonth() && 
                                  selectedDate.getFullYear() === now.getFullYear();

                    if (isToday) {
                        var currentHour = now.getHours();
                        if (currentHour < startHour) {
                            instance.set('minTime', startHour + ":00");
                        } else if (currentHour >= startHour && currentHour < endHour) {
                            instance.set('minTime', currentHour + ":" + now.getMinutes());
                        } else {
                            instance.set('minTime', endHour + ":00"); 
                        }
                    } else {
                        instance.set('minTime', startHour + ":00");
                    }
                    instance.set('maxTime', endHour + ":00");
                }
            }
        });
    </script>
    <?php
}
